/*
 * A super-simple database system
 */
#ifndef _db_h_
#define _db_h_

/*
 * types
 */
typedef struct db_database *db_database_ptr_t;
typedef struct db_table *db_table_ptr_t;

/* a datum */
typedef enum {
  DBD_INT,
  DBD_STR,
  DBD_DBL,
  DBD_MAC,
  DBD_MATCHALL,		/* used as a wildcard in queries */
  DBD_NUM_TYPES
} db_datum_type_t;

struct db_datum {
 db_datum_type_t dd_type;
 union value {
   int integer;
   char *string;
   double dbl_value;
   unsigned char mac_addr[6];
 } v;
};
typedef struct db_datum db_datum_t;

/* shortcuts for value access */
#define dbd_int v.integer
#define dbd_str v.string
#define dbd_dbl v.dbl_value
#define dbd_mac v.mac_addr

/* special datum value to match anything */
#define DB_DATUM_ANY ((db_datum_t *)1)

/*
 * A database row
 */
struct db_row {
  db_datum_t *datum;
};
typedef struct db_row *db_row_t;

/*
 * A database column
 */
struct db_column {
  db_datum_type_t type;
  char *name;
};
typedef struct db_column_t *db_column_t;

/*
 * Prototypes
 */
int db_create_database(	/* create an empty database */
  char *dir,
  char *db_name);		/* name of the database */

db_database_ptr_t db_open_database(	/* open an existing database */
  char *dir,
  char *db_name,
  int write);

int db_close_database(			/* close a database */
  db_database_ptr_t db);

int db_create_table(		/* create a table */
  db_database_ptr_t db,
  char *table_name,
  int ncols,
  char **col_names,
  db_datum_type_t *col_types);

db_table_ptr_t db_open_table(		/* get access to a table for ops */
  db_database_ptr_t db,
  char *table_name);
 
void db_close_table(			/* free resources for a table */
  db_table_ptr_t table);

int db_flush_table(			/* write table changes to disk */
  db_table_ptr_t tp);

int db_rename_table(			/* rename an open table */
  db_table_ptr_t tp,
  char *new_table_name);

int db_tbl_num_cols(			/* number of columns in a table */
  db_table_ptr_t tp);

int db_get_column_info(			/* get the column info for a name */
  db_table_ptr_t table,
  char *column_name,
  int *index,
  db_datum_type_t *type);

db_datum_t *db_alloc_row_template(	/* get column template for a table */
  db_table_ptr_t table,
  int *ncols);

int db_add_row(				/* add a row to a table */
  db_table_ptr_t table,
  db_datum_t *row);

int db_simple_query(			/* perform a very simple query */
  db_table_ptr_t table,
  int column,			/* column to match */
  db_datum_t *value,		/* value to look for */
  db_datum_t ***query_result,	/* query result */
  int *nres);			/* number of matches */

int db_simple_or_query(			/* perform a simple OR query */
  db_table_ptr_t table,
  int column,			/* column to match */
  db_datum_t *value,		/* values to look for */
  int num_vals,			/* number of values specified */
  db_datum_t ***query_result,	/* query result */
  int *nres);			/* number of matches */

void db_free_query_res(			/* free results of a query */
  db_table_ptr_t table,
  db_datum_t **qres,
  int nres);

int db_row_query(			/* perform a query on this table */
  db_table_ptr_t table,
  db_datum_t *query,		/* a row to match */
  db_datum_t ***query_result,	/* query result */
  int *nrows);			/* number of rows returned */

int db_row_or_query(			/* perform an OR query on this table */
  db_table_ptr_t table,
  db_datum_t **query,		/* a row to match */
  int num_rows,			/* number of rows to OR */
  db_datum_t ***query_result,	/* query result */
  int *nrows);			/* number of rows returned */

int db_delete_row(			/* delete all matching rows */
  db_table_ptr_t table,
  db_datum_t *pattern);		/* a row to match */

#endif /* _db_h_ */
